/*
 * Apple USB Touchpad (PowerBook5,6) driver
 *
 * Copyright (C) 2001-2004 Greg Kroah-Hartman (greg@kroah.com)
 * Copyright (C) 2005      Johannes Berg (johannes@sipsolutions.net)
 * Copyright (C) 2005      Stelian Pop (stelian@popies.net)
 *
 * Thanks to Alex Harper <basilisk@foobox.net> for his inputs.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/module.h>
#include <linux/usb.h>
#include <linux/input.h>

/* Apple has powerbooks which have the keyboard with different Product IDs */
#define APPLE_VENDOR_ID		0x05AC
#define ATP_12INCH_ID1		0x030A
#define ATP_15INCH_ID1		0x020E
#define ATP_15INCH_ID2		0x020F
#define ATP_17INCH_ID1		0x020D /* XXX need a tester !!! */

#define ATP_DRIVER_VERSION	0x0004 /* 00.04 */

#define ATP_DEVICE(prod)					\
	.match_flags = USB_DEVICE_ID_MATCH_DEVICE |   		\
		       USB_DEVICE_ID_MATCH_INT_CLASS |		\
		       USB_DEVICE_ID_MATCH_INT_PROTOCOL,	\
	.idVendor = APPLE_VENDOR_ID,				\
	.idProduct = (prod),					\
	.bInterfaceClass = 0x03,				\
	.bInterfaceProtocol = 0x02

/* table of devices that work with this driver */
static struct usb_device_id atp_table [] = {
	{ ATP_DEVICE(ATP_12INCH_ID1) },
	{ ATP_DEVICE(ATP_15INCH_ID1) },
	{ ATP_DEVICE(ATP_15INCH_ID2) },
	{ ATP_DEVICE(ATP_17INCH_ID1) },
	{ }					/* Terminating entry */
};
MODULE_DEVICE_TABLE (usb, atp_table);

/* size of a USB urb transfer */
#define ATP_DATASIZE	81

/*
 * number of sensors. Note that only 16 of the 26 x sensors are used on
 * 12" and 15" Powerbooks.
 */
#define ATP_XSENSORS	26
#define ATP_YSENSORS	16

/* amount of fuzz this touchpad generates */
#define ATP_FUZZ	16

/*
 * multiplication factor for the X and Y coordinates.
 * We try to keep the touchpad aspect ratio while still doing only simple
 * arithmetics.
 * The factors below give coordinates like:
 * 	0 <= x <  960 on 12" and 15" Powerbooks
 * 	0 <= x < 1600 on 17" Powerbooks
 * 	0 <= y =  480
 */
#define ATP_XFACT	64
#define ATP_YFACT	32

/*
 * Threshold for the touchpad sensors. Any change less than MIN_THRESHOLD is
 * ignored, and change over MAX_THRESHOLD is considered an sensor error.
 */
#define MIN_THRESHOLD	 4
#define MAX_THRESHOLD	48

/* maximum allowed consecutive invalid samples before resetting */
#define MAX_INVALID	 20

/* number of samples to read when resetting */
#define MAX_RESET	 10

/*
 * Maximum allowed change in x/y direction between two samples,
 * used in order to catch crazy points reported by the sensors
 * while warming.
 */
#define MAX_DELTA	 100

/* Structure to hold all of our device specific stuff */
struct atp {
	struct usb_device *	udev;		/* usb device */
	struct urb *		urb;		/* usb request block */
	unsigned char *		data;		/* transferred data */
	int			open;		/* open count */
	struct input_dev	input;		/* input dev */
};

#define dbg_dump(msg, tab) \
	if (debug > 1) {						\
		int i;							\
		printk("atp: %s ", msg);				\
		for (i = 0; i < ATP_XSENSORS + ATP_YSENSORS; i++)	\
			printk("%02x ", tab[i]); 			\
		printk("\n"); 						\
	}

#define dprintk(format, a...) 						\
	do {								\
		if (debug) printk(format, ##a);				\
	} while (0)

MODULE_AUTHOR("Johannes Berg, Stelian Pop");
MODULE_DESCRIPTION("Touchpad driver for Apple Powerbooks Alu (PowerBook5,6)");
MODULE_LICENSE("GPL");

static int debug = 1;
module_param(debug, int, 0644);
MODULE_PARM_DESC(debug, "Activate debugging output");

static int reset = MAX_RESET;
module_param(reset, int, 0644);
MODULE_PARM_DESC(reset, "Reset the touchpad levels");

static int atp_calculate_abs(int *xy_sensors, int nb_sensors, int fact,
			     int *detect) {
	int i;
	/* pressure width max, previous sensors and next sensor */
	int pmax = 0, pprev = 0, pnext = 0, psum;
	/* indexes of the first and last non zero sensors */
	int istart = -1, iend = -1;
	/* index of the sensor with the highest pressure */
	int imax = -1;

	*detect = 1;

	for (i = 0; i < nb_sensors; i++) {
		if (xy_sensors[i] > pmax) {
			pmax = xy_sensors[i];
			imax = i;
		}
		if (xy_sensors[i] != 0 && istart == -1)
			istart = i;
		if (xy_sensors[i] == 0 && istart != -1 && iend == -1)
			iend = i;
		if (xy_sensors[i] != 0 && iend != -1) {
			/*
			 * in the future, we could add here code to search for
			 * a second finger...
			 * for now, scrolling using the synaptics X driver is
			 * much more simpler to achieve.
			 */
			dprintk("atp: invalid sensor at %d (2 fingers ?)\n", i);
			return -1;
		}
	}

	if (istart == -1) {
		*detect = 0;	/* no pressure detected */
		return 0;
	}
	if (iend == -1)
		iend = nb_sensors;
	if (iend - istart > 5) {
		dprintk("atp: more than 5 consecutive sensors (big finger?)\n");
		return -1;
	}

	if (imax > 0)
		pprev = xy_sensors[imax - 1];
	if (imax < nb_sensors - 1)
		pnext = xy_sensors[imax + 1];
	psum = pprev + pmax + pnext;

	/* do we need a better linear transformation here ? */
	return imax * fact - pprev * fact / psum + pnext * fact / psum;
}

static void atp_complete(struct urb* urb, struct pt_regs* regs) {
	static int xy_zero[ATP_XSENSORS + ATP_YSENSORS];
	static int lastx = -1, lasty = -1;
	static int invalid = 0;
	int xy_sample[ATP_XSENSORS + ATP_YSENSORS];
	int xy_cur[ATP_XSENSORS + ATP_YSENSORS];
	int retval, i;
	int x_position, x_detect, y_position, y_detect;
	struct atp *dev = urb->context;

	switch (urb->status) {
	case 0:
		/* success */
		break;
	case -ECONNRESET:
	case -ENOENT:
	case -ESHUTDOWN:
		/* This urb is terminated, clean up */
		dbg("%s - urb shutting down with status: %d",
		    __FUNCTION__, urb->status);
	        return;
	default:
		dbg("%s - nonzero urb status received: %d",
		    __FUNCTION__, urb->status);
		goto exit;
	}

#define REORDER(which, offset) \
	for (i = 0; i < 8; i++) {\
		xy_sample[(offset) + i] = dev->data[5 * i + (which)];\
	}

	/* x axis */
	REORDER( 2,  0);
	REORDER( 4,  8);
	REORDER(42, 16);
	REORDER(44, 24);

	/* y axis */
	REORDER( 1, 26);
	REORDER( 3, 34);

	/* after MAX_INVALID invalid samples, reset the zero values */
	if (invalid > MAX_INVALID) {
		dprintk("atp: resetting.\n");
		invalid = 0;
		reset = MAX_RESET;
	}

	/* collect a few samples */
	if (reset > 0) {
		reset--;
		for (i = 0; i < ATP_XSENSORS + ATP_YSENSORS; i++)
			xy_zero[i] = xy_sample[i];
		dbg_dump("zero", xy_zero);
		goto exit;
	}

	dbg_dump("sample", xy_sample);

	for (i = 0; i < ATP_XSENSORS + ATP_YSENSORS; i++) {

		/* compute level */
		xy_cur[i] = xy_sample[i] - xy_zero[i];

		/* handle wraparound */
		if (xy_cur[i] > 127)
			xy_cur[i] -= 256;
		else if (xy_cur[i] < -127)
			xy_cur[i] += 256;

		/* ignore all levels less than THRESHOLD */
		if (abs(xy_cur[i]) > MAX_THRESHOLD) {
			dprintk("atp: invalid - value %d above MAX_THRESHOLD\n",
				xy_cur[i]);
			goto invalid;
		}
		else if (xy_cur[i] < -MIN_THRESHOLD) {
			dprintk("atp: invalid - negative value %d\n",
				xy_cur[i]);
			goto invalid;
		} else if (xy_cur[i] < MIN_THRESHOLD)
			xy_cur[i] = 0;
		else
			xy_cur[i] -= MIN_THRESHOLD;
	}

	dbg_dump("corrected sample", xy_cur);

	x_position = atp_calculate_abs(xy_cur, ATP_XSENSORS,
				       ATP_XFACT, &x_detect);
	if (x_position < 0)
		goto invalid;

	y_position = atp_calculate_abs(xy_cur + ATP_XSENSORS, ATP_YSENSORS,
				       ATP_YFACT, &y_detect);
	if (y_position < 0)
		goto invalid;

	if (x_detect && y_detect) {

		if (lastx != -1 &&
		    (abs(x_position - lastx) > MAX_DELTA ||
		     abs(y_position - lasty) > MAX_DELTA)) {
			dprintk("atp: invalid - too much change\n");
			goto invalid;
		}
		lastx = x_position;
		lasty = y_position;

		if (debug > 1)
			printk("atp: X: %3d Y: %3d\n", x_position, y_position);
		input_report_key(&dev->input, BTN_TOUCH, 1);
		input_report_abs(&dev->input, ABS_X, x_position);
		input_report_abs(&dev->input, ABS_Y, y_position);
		input_report_abs(&dev->input, ABS_PRESSURE, 100);
		input_report_key(&dev->input, BTN_TOOL_FINGER, 1);
	}
	else if (!x_detect && !y_detect) {

		lastx = lasty = -1;

		input_report_key(&dev->input, BTN_TOUCH, 0);
		input_report_abs(&dev->input, ABS_PRESSURE, 0);
	}
	else {
		dprintk("atp: invalid only x or y change.\n");
		goto invalid;
	}

	invalid = 0;

	input_report_key(&dev->input, BTN_LEFT, !!dev->data[80]);

	input_sync(&dev->input);

invalid:
	invalid++;
exit:
	retval = usb_submit_urb(dev->urb, GFP_ATOMIC);
	if (retval) {
		err("%s - usb_submit_urb failed with result %d",
		    __FUNCTION__, retval);
	}
}

static int atp_open(struct input_dev *input)
{
	struct atp *dev = input->private;

        if (dev->open++)
                return 0;

	if (usb_submit_urb(dev->urb, GFP_ATOMIC)) {
		dev->open--;
		return -EIO;
	}

	return 0;
}

static void atp_close(struct input_dev *input)
{
	struct atp *dev = input->private;

	if (!--dev->open)
		usb_kill_urb(dev->urb);
}

static int atp_probe(struct usb_interface *iface, const struct usb_device_id *id)
{
	struct atp *dev = NULL;
	struct usb_host_interface *iface_desc;
	struct usb_endpoint_descriptor *endpoint;
	int int_in_endpointAddr = 0;
	int i, retval = -ENOMEM;

	/* allocate memory for our device state and initialize it */
	dev = kmalloc(sizeof(struct atp), GFP_KERNEL);
	if (dev == NULL) {
		err("Out of memory");
		goto err_kmalloc;
	}
	memset(dev, 0, sizeof(struct atp));

	dev->udev = interface_to_usbdev(iface);

	/* set up the endpoint information */
	/* use only the first interrupt-in endpoint */
	iface_desc = iface->cur_altsetting;
	for (i = 0; i < iface_desc->desc.bNumEndpoints; i++) {
		endpoint = &iface_desc->endpoint[i].desc;
		if (!int_in_endpointAddr &&
                    (endpoint->bEndpointAddress & USB_DIR_IN) &&
		    ((endpoint->bmAttributes & USB_ENDPOINT_XFERTYPE_MASK)
					== USB_ENDPOINT_XFER_INT)) {
			/* we found an interrupt in endpoint */
			int_in_endpointAddr = endpoint->bEndpointAddress;
			break;
		}
	}
	if (!int_in_endpointAddr) {
		retval = -EIO;
		err("Could not find int-in endpoint");
		goto err_endpoint;
	}

	/* save our data pointer in this interface device */
	usb_set_intfdata(iface, dev);

	dev->urb = usb_alloc_urb(0, GFP_KERNEL);
	if (!dev->urb) {
		retval = -ENOMEM;
		goto err_usballoc;
	}
	dev->data = usb_buffer_alloc(dev->udev, ATP_DATASIZE, GFP_KERNEL,
				     &dev->urb->transfer_dma);
	if (!dev->data) {
		retval = -ENOMEM;
		goto err_usbbufalloc;
	}
	usb_fill_int_urb(dev->urb, dev->udev,
			 usb_rcvintpipe(dev->udev, int_in_endpointAddr),
			 dev->data, ATP_DATASIZE, atp_complete, dev, 1);

	init_input_dev(&dev->input);
	dev->input.name = "atp";
	dev->input.dev = &iface->dev;
	dev->input.private = dev;
	dev->input.open = atp_open;
	dev->input.close = atp_close;

	dev->input.id.bustype = BUS_USB;
	dev->input.id.vendor = id->idVendor;
	dev->input.id.product = id->idProduct;
	dev->input.id.version = ATP_DRIVER_VERSION;

	set_bit(EV_ABS, dev->input.evbit);
	if (id->idProduct == ATP_17INCH_ID1)
		input_set_abs_params(&dev->input, ABS_X, 0,
				     (ATP_XSENSORS - 1) * ATP_XFACT - 1,
				     ATP_FUZZ, 0);
	else
		/* 12" and 15" Powerbooks only have 16 x sensors */
		input_set_abs_params(&dev->input, ABS_X, 0,
				     (16 - 1) * ATP_XFACT - 1,
				     ATP_FUZZ, 0);
	input_set_abs_params(&dev->input, ABS_Y, 0,
			     (ATP_YSENSORS - 1 ) * ATP_YFACT - 1,
			     ATP_FUZZ, 0);
	input_set_abs_params(&dev->input, ABS_PRESSURE, 0, 100, 0, 0);

	set_bit(EV_KEY, dev->input.evbit);
	set_bit(BTN_TOUCH, dev->input.keybit);
	set_bit(BTN_TOOL_FINGER, dev->input.keybit);
	set_bit(BTN_LEFT, dev->input.keybit);

	input_register_device(&dev->input);

	printk(KERN_INFO "input: atp connected\n");

	return 0;

err_usbbufalloc:
	usb_free_urb(dev->urb);
err_usballoc:
	usb_set_intfdata(iface, NULL);
err_endpoint:
	kfree(dev);
err_kmalloc:
	return retval;
}

static void atp_disconnect(struct usb_interface *iface)
{
	struct atp *dev = usb_get_intfdata(iface);

	usb_set_intfdata(iface, NULL);
	if (dev) {
		usb_kill_urb(dev->urb);
		input_unregister_device(&dev->input);
		usb_free_urb(dev->urb);
		usb_buffer_free(dev->udev, ATP_DATASIZE,
				dev->data, dev->urb->transfer_dma);
		kfree(dev);
	}
	printk(KERN_INFO "input: atp disconnected\n");
}

static int atp_suspend(struct usb_interface *iface, pm_message_t message)
{
	struct atp *dev = usb_get_intfdata(iface);
	if (dev->open)
		usb_kill_urb(dev->urb);
	return 0;
}

static int atp_resume(struct usb_interface *iface)
{
	struct atp *dev = usb_get_intfdata(iface);
	if (dev->open && usb_submit_urb(dev->urb, GFP_ATOMIC)) {
		dev->open--;
		return -EIO;
	}

	return 0;
}

static struct usb_driver atp_driver = {
	.owner		= THIS_MODULE,
	.name		= "atp",
	.probe		= atp_probe,
	.disconnect	= atp_disconnect,
	.suspend	= atp_suspend,
	.resume		= atp_resume,
	.id_table	= atp_table,
};

static int __init atp_init(void)
{
	return usb_register(&atp_driver);
}

static void __exit atp_exit(void)
{
	usb_deregister(&atp_driver);
}

module_init (atp_init);
module_exit (atp_exit);
