/*
 *  Envlink File System, version 0.2, 27.04.1998
 *
 *  envlinkfs/init.c
 *
 *  Copyright (C) 1998 by Stelian Pop <pop@cybercable.fr>
 *
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/malloc.h>
#include <asm/segment.h>
#include <linux/locks.h>

#include "envlinkfs.h"

extern struct inode_operations envlinkfs_inode_operations;
extern struct inode_operations envlinkfs_dir_inode_operations;

/*
 * Fills the inode structure.
 */
static void 
envlinkfs_read_inode(struct inode *inode)
{
	unsigned long ino;

	PRINTK("envlinkfs: envlinkfs_read_inode\n");
	inode->i_op = NULL;
        inode->i_mode = 0;
        inode->i_uid = 0;
        inode->i_gid = 0;
        inode->i_nlink = 1;
        inode->i_size = 0;
        inode->i_mtime = inode->i_atime = inode->i_ctime = CURRENT_TIME;
        inode->i_blocks = 0;
        inode->i_blksize = 1024;

	ino = inode->i_ino;
	switch (ino) {
		case ENVLINKFS_ROOT_INODE:
			inode->i_mode = S_IFDIR | S_IRUGO | S_IXUGO;
	                inode->i_nlink = 2;
			inode->i_op = &envlinkfs_dir_inode_operations;
			return;
		default:
			inode->i_mode = S_IFLNK | S_IRUGO | S_IWUGO | S_IXUGO;
			inode->i_op = &envlinkfs_inode_operations;
			return;
	}
	return;
}

/*
 * The filesystem is read-only, so the inode will not be written.
 */
static void
envlinkfs_write_inode(struct inode *inode)
{
	PRINTK("envlinkfs: envlinkfs_write_inode\n");

	inode->i_dirt=0;
}

/*
 * Free the inode.
 */
static void 
envlinkfs_put_inode(struct inode *inode)
{
	PRINTK("envlinkfs: envlinkfs_put_inode\n");

        if (inode->i_nlink)
                return;
        inode->i_size = 0;
}

/*
 * Free the superblock.
 */
static void
envlinkfs_put_super(struct super_block *sb)
{
        PRINTK("envlinkfs: envlinkfs_put_super\n");
        lock_super(sb);
        sb->s_dev = 0;
        unlock_super(sb);
        MOD_DEC_USE_COUNT;
}

/*
 * File system statistics.
 */
static void 
envlinkfs_statfs(struct super_block *sb, struct statfs *buf, int bufsiz)
{
        struct statfs tmp;

	PRINTK("envlinkfs: envlinkfs_statfs\n");

        tmp.f_type = ENVLINKFS_SUPER_MAGIC;
        tmp.f_bsize = PAGE_SIZE/sizeof(long);
        tmp.f_blocks = 0;
        tmp.f_bfree = 0;
        tmp.f_bavail = 0;
        tmp.f_files = 0;
        tmp.f_ffree = 0;
        tmp.f_namelen = NAME_MAX;
        memcpy_tofs(buf, &tmp, bufsiz);
}

/*
 * Superblock operations.
 */
static struct super_operations envlinkfs_sops = {
	envlinkfs_read_inode,
	NULL,
	envlinkfs_write_inode,
	envlinkfs_put_inode,
	envlinkfs_put_super,
	NULL,
	envlinkfs_statfs,
	NULL
};

/*
 * Returns the inode referenced by its number.
 */
struct inode *
envlinkfs_get_inode(struct super_block *s, int ino)
{
	struct inode *inode;

	PRINTK("envlinkfs: envlinkfs_get_inode\n");
	inode = iget(s, ino);
	if (inode && inode->i_sb == s)
		envlinkfs_read_inode(inode);
	return inode;
}

/*
 * Initialise the superblock.
 */
struct super_block *
envlinkfs_read_super(struct super_block *s, void *data, int silent)
{
        PRINTK("envlinkfs: read_super\n");

	MOD_INC_USE_COUNT;
        lock_super(s);
        s->s_blocksize = 1024;
        s->s_blocksize_bits = 10;
        s->s_magic = ENVLINKFS_SUPER_MAGIC;
        s->s_op = &envlinkfs_sops;
	unlock_super(s);
	s->s_mounted = envlinkfs_get_inode(s, ENVLINKFS_ROOT_INODE);
        return s;
}

static struct file_system_type envlinkfs_fs_type = {
	envlinkfs_read_super, "envlinkfs", 0, NULL
};

/*
 * Insmod-time options.
 */
char *options=NULL;

/*
 * Chained list of the directory entries.
 */
struct envlinkfs_dir_entry *de=NULL;

/*
 * Parses the options line and fills 'de'.
 */
int 
envlinkfs_read_options(void) {

	char *ptr1, *ptr2;
	struct envlinkfs_dir_entry *entry;
	unsigned int inode = ENVLINKFS_FIRST_INODE;

	if (!options)
		return -1;

	PRINTK("envlinkfs: envlinkfs_read_options: %s\n", options);

	ptr1 = strtok(options, " \t");
	if (!ptr1)
		return -1;
	ptr2 = strtok(NULL, " \t");
	if (!ptr2)
		return -1;

	while (1) {
		entry = (struct envlinkfs_dir_entry *)kmalloc(sizeof(struct envlinkfs_dir_entry), GFP_KERNEL);
		entry->ino = inode++;
		entry->name = kmalloc(strlen(ptr1)+1, GFP_KERNEL);
		if (*ptr1 == '$') {
			strcpy(entry->name, ptr1+1);
			entry->name_is_env = 1;
		}
		else {
			strcpy(entry->name, ptr1);
			entry->name_is_env = 0;
		}
		entry->dest = kmalloc(strlen(ptr2)+1, GFP_KERNEL);	
		if (*ptr2 == '$') {
			strcpy(entry->dest, ptr2+1);
			entry->dest_is_env = 1;
		}
		else {
			strcpy(entry->dest, ptr2);
			entry->dest_is_env = 0;
		}
		entry->next = de;
		de = entry;

		PRINTK("envlinkfs: added translation %s (%d) -> %s\n", ptr1, inode - 1, ptr2);

		ptr1 = strtok(NULL, " \t");
		if (!ptr1)
			return 0;
		ptr2 = strtok(NULL, " \t");
		if (!ptr2)
			return 0;
	}
	return 0;
}

/*
 * Free 'de'.
 */
void
envlinkfs_free(void)
{
	struct envlinkfs_dir_entry *entry;

	PRINTK("envlinkfs: envlinkfs_free\n");	
	while (de) {

		entry = de->next;
		kfree(de->name);
		kfree(de->dest);
		kfree(de);
		de = entry;
	}
}

/*
 * Initialisation of the module.
 */
int
init_module(void)
{
	int err;

	PRINTK("envlinkfs: init_module\n");

	err = envlinkfs_read_options();
	if (err) {
		PRINTK("envlinkfs: incorrect options\n");
		return err;
	}

	err = register_filesystem(&envlinkfs_fs_type);
	if (err) {
		PRINTK("envlinkfs: failed to register filesystem\n");
		return err;
	}

	register_symtab(0);

	return 0;
}

/*
 * Cleanup of the module.
 */
void 
cleanup_module(void)
{
	int err;

	PRINTK("envlinkfs: cleanup_module\n");

	envlinkfs_free();

	err = unregister_filesystem(&envlinkfs_fs_type);
	if (err) 
		PRINTK("envlinkfs: failed to unregister filesystem\n");
}
