/*
 *  Envlink File System, version 0.3, 01.02.1999
 *
 *  envlinkfs/inode.c
 *
 *  Copyright (C) 1998-1999 by Stelian Pop <pop@cybercable.fr>
 *
 */

#include <linux/config.h>
#include <linux/version.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/errno.h>
#include <linux/malloc.h>
#include <asm/segment.h>
#include <asm/uaccess.h>

#include "envlinkfs.h"

extern int envlinkfs_getenv_if(char *, char *, int, int);
extern int envlinkfs_envexists_if(char *, int);
extern struct envlinkfs_dir_entry *de;

/*
 * Returns the name of the link.
 */
static int 
envlinkfs_inode_readlink(struct dentry *dentry, char *buffer, int buflen)
{
	int len;
	char *name;
	struct envlinkfs_dir_entry *entry = de;
	struct inode *inode = dentry->d_inode;

	PRINTK(KERN_DEBUG "envlinkfs: envlinkfs_inode_readlink\n");

	if (!inode)
		return -ENOENT;

	while (entry) {
		if (entry -> ino == inode -> i_ino)
			break;
		entry = entry -> next;
	}
	if (!entry)
		return -ENOENT;

	name = kmalloc(4096, GFP_KERNEL); 
	if ((envlinkfs_getenv_if(entry->dest, name, 4096, entry->dest_is_env) == -1)
		|| (envlinkfs_envexists_if(entry->name, entry->name_is_env) == -1)) {
		kfree(name);
		return -ENOENT;
	}
	
	if (strlen(name) < buflen)
		len = strlen(name);
	else
		len = buflen;
	copy_to_user(buffer, name, len);
	kfree(name);
	return len;
}

/*
 * Returns the destination of the link.
 */
static struct dentry *
envlinkfs_inode_followlink(struct dentry *dentry, struct dentry *base, unsigned int follow) 
{
	char *name;
	struct envlinkfs_dir_entry *entry = de;
	struct inode *inode = dentry->d_inode;

	PRINTK(KERN_DEBUG "envlinkfs: envlinkfs_inode_followlink\n");

	while (entry) {
		if (entry -> ino == inode -> i_ino)
			break;
		entry = entry -> next;
	}
	if (!entry) {
		dput(base);
		return ERR_PTR(-EAGAIN);
	}

	name = kmalloc(4096, GFP_KERNEL);
	if  ( (envlinkfs_getenv_if(entry->dest, name, 4096,entry->dest_is_env) == -1) || (envlinkfs_envexists_if(entry->name, entry->name_is_env) == -1) ) {
		dput(base); 
		kfree(name);
		return ERR_PTR(-EAGAIN);
	}

	dentry = lookup_dentry(name, base, follow);
	kfree(name);
	return dentry;
}

/*
 * Inode operations: simple, we have only envlinks.
 */
struct inode_operations envlinkfs_inode_operations = {
	NULL,                   /* no */
	NULL,                   /* create */
	NULL,                   /* lookup */
	NULL,                   /* link */
	NULL,                   /* unlink */
	NULL,                   /* envlink */
	NULL,                   /* mkdir */
	NULL,                   /* rmdir */
	NULL,                   /* mknod */
	NULL,                   /* rename */
	envlinkfs_inode_readlink, /* readlink */
	envlinkfs_inode_followlink,/* follow_link */
	NULL,                   /* readpage */
	NULL,                   /* writepage */
	NULL,                   /* bmap */
	NULL,                   /* truncate */
	NULL                    /* permissions */
};

