/*
 *  Envlink File System, version 0.3, 01.02.1999
 *
 *  envlinkfs/root.c
 *
 *  Copyright (C) 1998-1999 by Stelian Pop <pop@cybercable.fr>
 *
 */

#include <linux/config.h>
#include <linux/version.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/errno.h>
#include <linux/malloc.h>
#include <asm/segment.h>

#include "envlinkfs.h"

extern struct inode *envlinkfs_get_inode(struct super_block *, int);
extern int envlinkfs_getenv_if(char *, char *, int, int);
extern int envlinkfs_envexists_if(char *, int);
extern struct envlinkfs_dir_entry *de;

/*
 * Returns the directory content.
 */
static int 
envlinkfs_dir_readdir(struct file *filp, void *dirent, filldir_t filldir) 
{
	unsigned int ino;
	int i;
	char *name;
	struct envlinkfs_dir_entry *entry = de;
	struct inode *inode = filp->f_dentry->d_inode;

	PRINTK(KERN_DEBUG "envlinkfs: envlinkfs_dir_readdir\n");

	if (!inode || !S_ISDIR(inode->i_mode))
		return -ENOTDIR;
	ino = inode->i_ino;
	i = filp->f_pos;

	switch(i) {
		case 0:
			if (filldir(dirent, ".", 1, i, ino) < 0)
				return 0;
			i++;
			filp->f_pos++;
			/* fall through */
		case 1:
			if (filldir(dirent, "..", 2, i, ENVLINKFS_ROOT_INODE) < 0)
				return 0;
			i++;
			filp->f_pos++;
		default:
			while (entry 
				&& ( envlinkfs_envexists_if(entry->name, entry->name_is_env)==-1 
					|| envlinkfs_envexists_if(entry->dest, entry->dest_is_env)==-1 ) )
				entry = entry -> next;

			i -= 2;
			for (;;) {
				if (!entry)
					return 1;
				if (!i)
					break;
				entry = entry -> next;
				while (entry
					&& ( envlinkfs_envexists_if(entry->name, entry->name_is_env)==-1
						|| envlinkfs_envexists_if(entry->dest, entry->dest_is_env)==-1 ) )
					entry = entry -> next;
				i--;
			}

			name = kmalloc(4096, GFP_KERNEL); 
			do {	
				if ((envlinkfs_getenv_if(entry->name, name, 4096, entry->name_is_env) != -1)
					&& (envlinkfs_envexists_if(entry->dest, entry->dest_is_env) != -1)) {
					if (filldir(dirent, name, strlen(name), filp->f_pos, entry->ino) < 0) {
						kfree(name);
						return 0;
					}
					filp->f_pos++;
				}
				entry = entry -> next;
			} while (entry);
	}
	kfree(name);
	return 0;
}

/*
 * Finds the inode of an entry in the directory.
 */
static int 
envlinkfs_dir_lookup(struct inode *dir, struct dentry *dentry) 
{
	unsigned int ino;
	char *nameenv;
	struct envlinkfs_dir_entry *entry = de;
	char *name;
	int len;
	struct inode *result = NULL;

	PRINTK(KERN_DEBUG "envlinkfs: envlinkfs_dir_lookup: %s\n", dentry->d_name.name);

	if (!dir || !S_ISDIR(dir->i_mode))
		return -ENOTDIR;

	ino = dir->i_ino;

	result = dir;
	name = (char *)dentry->d_name.name;
	len = dentry->d_name.len;
	if (!len) {
		d_add(dentry, result);
		return 0;
	}

	/* Special case "." and ".." points on the same dir */
	if (name[0] == '.') {
		if (len == 1) {
			d_add(dentry, result);
			return 0;
		}
		if (name[1] == '.' && len == 2) {
			d_add(dentry, result);
			return 0;
		}
	}

	nameenv = kmalloc(4096, GFP_KERNEL);

	while (entry) {
		if ((envlinkfs_getenv_if(entry->name, nameenv, 4096, entry->name_is_env) != -1)
			&& (envlinkfs_envexists_if(entry->dest, entry->dest_is_env) != -1)) {
			if ((len==strlen(nameenv)) && (!memcmp(name, nameenv, len))) {
				if (!(result = envlinkfs_get_inode(dir->i_sb, entry->ino))) {
					kfree(nameenv);
					return -EINVAL;
				}
				d_add(dentry, result); 
				kfree(nameenv);
				return 0;
			}
		}
		entry = entry -> next;
	}
	kfree(nameenv);
	return -EINVAL; 
}

/*
 * Directory file operations.
 */
static struct file_operations envlinkfs_dir_operations = {
	NULL,                   /* lseek - default */
	NULL,                   /* read */
	NULL,                   /* write - bad */
	envlinkfs_dir_readdir,    /* readdir */
	NULL,                   /* select - default */
	NULL,                   /* ioctl - default */
	NULL,                   /* mmap */
	NULL,                   /* no special open code */
	NULL,			/* flush */
	NULL,                   /* no special release code */
	NULL,                   /* can't fsync */
	NULL,			/* fasync */
	NULL,			/* check media change */
	NULL			/* revalidate */
};

/*
 * Directory inode operations.
 */
struct inode_operations envlinkfs_dir_inode_operations = {
	&envlinkfs_dir_operations,   /* default directory file-ops */
	NULL,                   /* create */
	envlinkfs_dir_lookup,     /* lookup */
	NULL,                   /* link */
	NULL,                   /* unlink */
	NULL,                   /* envlink */
	NULL,                   /* mkdir */
	NULL,                   /* rmdir */
	NULL,                   /* mknod */
	NULL,                   /* rename */
	NULL,                   /* readlink */
	NULL,                   /* follow_link */
	NULL,                   /* readpage */
	NULL,                   /* writepage */
	NULL,                   /* bmap */
	NULL,                   /* truncate */
	NULL,                   /* permissions */
	NULL			/* smap */
};

