/*
 * kissd - KiSS PC-Link Daemon
 *
 * Copyright (C) 2005 Stelian Pop <stelian@popies.net>
 *
 * Heavily based on kiss4lin,
 * Copyright (C) 2004 Jacob Kolding <dacobi@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <dirent.h>
#include <limits.h>
#include <libgen.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <sys/wait.h>

#ifdef USE_INTERNAL_SENDFILE
#include <sys/mman.h>
#else
#ifdef Linux
#include <sys/sendfile.h>
#endif
#endif

#include "kissd.h"

static int store_recent(char * filename);

/*
 *   Name: timesort
 *
 *   Description: Sort function for scandir which sorts by time.
 *                Most recent files first.
 */
static int timesort(const void * f1, const void * f2)
{
	char fname1[PATH_MAX];
	char fname2[PATH_MAX];
	char cwd[PATH_MAX];
	struct stat st1, st2;
	struct dirent *d1;
	struct dirent *d2;

	if (f1 == NULL || f2 == NULL)
		return 0;

	d1 = *(struct dirent **)f1;
	d2 = *(struct dirent **)f2;

	if (getcwd(cwd, PATH_MAX) == NULL)
		return 0;

	snprintf(fname1, PATH_MAX, "%s/%s", cwd, d1->d_name);
	snprintf(fname2, PATH_MAX, "%s/%s", cwd, d2->d_name);

	if (lstat(fname1, &st1) < 0)
		log("Couldn't stat %s", fname1);
	if (lstat(fname2, &st2) < 0)
		log("Couldn't stat %s", fname2);

	if (st1.st_ctime > st2.st_ctime)
		return -1;  /* if timestamp is bigger, file is newer */
	if (st1.st_ctime < st2.st_ctime)
		return 1;   /* if timestamp is smaller, file is older */

	return 0;
}

int run_trigger(char *command, char *filename, size_t fsize, char *result, size_t size) {
	int pid, status;
	int p[2];
	size_t len;

	if (!command[0]) {
		if (result)
			memcpy(result, filename, min(fsize, size));
		return 0;
	}

	if (pipe(p) < 0) {
		log("trigger pipe() failed: %s", strerror(errno));
		return -1;
	}

	pid = fork();
	if (pid < 0) {
		log("trigger fork() failed: %s", strerror(errno));
		close(p[0]);
		close(p[1]);
		return -1;
	}
	if (pid == 0) {
		setpgid(pid, 0);
		close(p[0]);
		close(STDOUT_FILENO);
		if (dup2(p[1], STDOUT_FILENO) < 0) {
			log("trigger dup2() failed: %s", strerror(errno));
			exit(1);
		}
		execlp(command, command, filename, NULL);
		log("trigger exec() failed: %s", strerror(errno));

		exit(1);
	}
	close(p[1]);

	if (waitpid(pid, &status, 0) < 0) {
		log("trigger waitpid() failed: %s", strerror(errno));
		close(p[0]);
		return -1;
	}

	if (!WIFEXITED(status)) {
		log("trigger child exited abnormally: %x", status);
		close(p[0]);
		return -1;
	}

	if (WEXITSTATUS(status)) {
		log("trigger child returned non 0: %d", WEXITSTATUS(status));
		close(p[0]);
		return -1;
	}

	if (result) {
		if ((len = read(p[0], result, size)) < 0) {
			log("trigger read() failed: %s", strerror(errno));
			close(p[0]);
			return -1;
		}
		while (len > 0 && (result[len - 1] == '\r' || result[len - 1] == '\n'))
			len--;
		result[len < size ? len : len - 1] = '\0';
	}

	close(p[0]);

	return 0;
}

static int do_recv(int sock, char *buffer, int size) {
	int len;

	if ((len = recv(sock, buffer, size, 0)) < 0) {
		log("recv: %s", strerror(errno));
		return len;
	}
	if (len == 0)
		return -1;
	while (len > 0 && (buffer[len - 1] == '\r' || buffer[len - 1] == '\n'))
		len--;
	/* properly end the string and prevent buffer overflows */
	buffer[len < size ? len : len - 1] = '\0';

	logv("<-- [%s]", buffer);

	return len;
}

int do_send(int sock, char *buffer, int size) {
	int len;

	logv("--> [%s]", buffer);

	if ((len = send(sock, buffer, size, 0)) < 0)
		log("send: %s", strerror(errno));
	return len;
}

/*
 * Name: clean_pathname
 *
 * Description: Replaces unsafe/incorrect instances of:
 *  //[...] with /
 *  /./ with /
 *  /../ with / (technically not what we want, but browsers should deal
 *   with this, not servers)
 */
void clean_pathname(char *pathname)
{
	char *cleanpath, c;

	cleanpath = pathname;
	while ((c = *pathname++)) {
		if (c == '/') {
			while (1) {
				if (*pathname == '/')
					pathname++;
				else if (*pathname == '.' && *(pathname + 1) == '/')
					pathname += 2;
				else if (*pathname == '.' && *(pathname + 1) == '.' &&
					*(pathname + 2) == '/') {
					pathname += 3;
				} else
					break;
			}
		}
		*cleanpath++ = c;
	}
	*cleanpath = '\0';
}

static int verify_path(char *p)
{
	char path[PATH_MAX];

	snprintf(path, PATH_MAX, "/%s", p);
	clean_pathname(path);
	if (strstr(path, audiopath) == path ||
	    strstr(path, videopath) == path ||
	    strstr(path, picturepath) == path)
		return 0;
	return -1;
}

static void list_folder(int sock, char *base, char *path)
{
	char fullpath[PATH_MAX], line[PATH_MAX];
	int i, n, to_send;
	struct stat st;
	struct dirent **namelist;
	int (*sortmethod)(const void *,const void *) = alphasort;

	if (strstr(path, "Recently used")) {
		path = ".recent";
		sortmethod = timesort;
	}

	snprintf(fullpath, PATH_MAX, "%s/%s", base, path);
	clean_pathname(fullpath);
	if (verify_path(fullpath) < 0) {
		log("access denied: %s", fullpath);
		return;
	}

	if (strstr(path, ".m3u\0") || strstr(path, ".pls\0") ) {
		handle_playlist(sock, fullpath);
		return;
	}

	chdir(fullpath); /* scandir does not provide path info, so set it */
	if ((n = scandir(fullpath, &namelist, 0, sortmethod)) < 0) {
		log("scandir %s: %s", fullpath, strerror(errno));
		do_send(sock, "EOL\n", 4); /* avoid a hanging kiss here */
		return;
	}

	/*
	 * do this only in root and only if no -a option set
	 * to avoid duplicate entry
	 */
	if (!a_opt && path[0] == '\0')
		if ((to_send = snprintf(line, PATH_MAX,
					"Recently used|Recently used|1|\n")) < PATH_MAX)
			do_send(sock, line, to_send);

	for (i = 0; i < n; i++) {
		char fpath[PATH_MAX];
		snprintf(fpath, PATH_MAX, "%s/%s/%s", base, path, namelist[i]->d_name);
		clean_pathname(fpath);
		if (stat(fpath, &st) < 0) {
			log("stat %s: %s", fpath, strerror(errno));
			continue;
		}
		if (!a_opt && namelist[i]->d_name[0] == '.')
			continue;

		if (S_ISDIR(st.st_mode)) {
			if ((strcmp(namelist[i]->d_name, ".") == 0) ||
			    (strcmp(namelist[i]->d_name, "..") == 0))
				continue;
			if ((to_send = snprintf(line, PATH_MAX, "%s|%s|1|\n",
						namelist[i]->d_name,
						namelist[i]->d_name)) < PATH_MAX)
				do_send(sock, line, to_send);
		}
		else if (strstr(namelist[i]->d_name, ".m3u\0") ||
			 strstr(namelist[i]->d_name, ".pls\0")) {
			if ((to_send = snprintf(line, PATH_MAX, "%s|%s|1|\n",
						namelist[i]->d_name,
						namelist[i]->d_name)) < PATH_MAX)
				do_send(sock, line, to_send);
		}
	}

	for (i = 0; i < n; i++) {
		char fpath[PATH_MAX];
		snprintf(fpath, PATH_MAX, "%s/%s/%s", base, path, namelist[i]->d_name);
		clean_pathname(fpath);
		if (stat(fpath, &st) < 0) {
			log("stat %s: %s", fpath, strerror(errno));
			free(namelist[i]);
			continue;
		}
		if (!a_opt && namelist[i]->d_name[0] == '.') {
			free(namelist[i]);
			continue;
		}

		if (! S_ISDIR(st.st_mode)) {
			if ((to_send = snprintf(line, PATH_MAX, "%s|%s|0|\n",
						namelist[i]->d_name, fpath)) < PATH_MAX)
				do_send(sock, line, strlen(line));
		}
		free(namelist[i]);
	}

	free(namelist);

	do_send(sock, "EOL\n", 4);
}

static void handle_list(int sock, char *request, size_t reqsize)
{
	char *p1, *p2;

	p1 = index(request, '|');
	p2 = rindex(request, '|');

	if (p1 == NULL || p2 == NULL) {
		log("invalid LIST command \"%s\"", request);
		return;
	}
	*p2 = '\0';

	if (strstr(request, "AUDIO") == request + 5)
		list_folder(sock, audiopath, p1 + 1);
	else if (strstr(request, "VIDEO") == request + 5)
		list_folder(sock, videopath, p1 + 1);
	else if (strstr(request, "PICTURE") == request + 5)
		list_folder(sock, picturepath, p1 + 1);
	else
    		log("unknown LIST command \"%s\"", request);
}

static void handle_size(int sock, char *request, size_t reqsize)
{
	struct stat st;
	char response[20];
	char *p1;
	char tfilename[PATH_MAX];

	p1 = index(request, '|');
	if (p1 == NULL) {
		log("invalid SIZE command \"%s\"", request);
		return;
	}
	*p1 = '\0';

	if (verify_path(request + 5) < 0) {
		log("access denied: %s", request + 5);
		return;
	}

	if (run_trigger(pretrigger, request + 5, reqsize - 5, tfilename, sizeof(tfilename)) < 0) {
		log("pretrigger %s failed", tfilename);
		return;
	}

	if (stat(tfilename, &st) < 0) {
		log("stat %s: %s", tfilename, strerror(errno));
      		return;
	}

 	snprintf(response, sizeof(response), "%015lld", (long long)st.st_size);
  	do_send(sock, response, 15);

	run_trigger(posttrigger, tfilename, sizeof(tfilename), NULL, 0);
}

static void handle_get(int sock, char *request, size_t reqsize, int fd)
{
	char *filename;
	char *p1, *p2;
	off_t offset;
	size_t chunk;
	int lfd = -1;
	long long o;
	long c;
	char tfilename[PATH_MAX];
#if defined(Linux) || defined(USE_INTERNAL_SENDFILE)
	size_t sent;
#else /* FreeBSD */
	off_t sent;
#endif

	p1 = index(request, '/');
	p2 = index(request, '|');
	if (p1 == NULL || p2 == NULL) {
		log("invalid GET command \"%s\"", request);
		return;
	}
	*p2 = '\0';

	filename = p1;
	if (sscanf(p2 + 1, "%lld %ld", &o, &c) != 2) {
		log("invalid GET parameters \"%s\"", p2 + 1);
		return;
	};
	offset = o;
	chunk = c;

	if (fd == -1) {

		if (verify_path(filename) < 0) {
			log("access denied: %s", filename);
			return;
		}

		if (run_trigger(pretrigger, filename, reqsize - (filename - request), tfilename, sizeof(tfilename)) < 0) {
			log("pretrigger %s failed", filename);
			return;
		}
		filename = tfilename;

		if ((lfd = open(filename, O_RDONLY)) < 0) {
			log("open %s: %s", filename, strerror(errno));
			return;
		}
		fd = lfd;

		store_recent(filename);
	}

	if (chunk == 0) {
		struct stat st;

		if (stat(filename, &st) < 0) {
			log("stat %s: %s", filename, strerror(errno));
      			if (lfd != -1) {
				run_trigger(posttrigger, filename, reqsize - (filename - request), NULL, 0);
				close(lfd);
			}
      			return;
		}
		chunk = st.st_size;
	}

#ifdef USE_INTERNAL_SENDFILE
	if ((sent = internal_sendfile(sock, fd, &offset, chunk)) < 0) {
#else
#ifdef Linux
	if ((sent = sendfile(sock, fd, &offset, chunk)) < 0) {
#else /* FreeBSD */
	if (sendfile(fd, sock, offset, chunk, NULL, &sent, 0) < 0) {
#endif
#endif
		log("sendfile: %s", strerror(errno));
		if (lfd != -1) {
			run_trigger(posttrigger, filename, reqsize - (filename - request), NULL, 0);
			close(lfd);
		}
		return;
	}

	/* The client asked for more data than we have, send padding bytes */
	while (sent++ < chunk)
		write(sock, "\0", 1);

	if (lfd != -1) {
		run_trigger(posttrigger, filename, reqsize - (filename - request), NULL, 0);
		close(lfd);
	}
}

static void handle_action1(int sock, char *request, size_t reqsize)
{
	char *p1, *p2;
	int fd;
	int len;
	char tfilename[PATH_MAX];

	p1 = index(request, '/');
	p2 = rindex(request, '|');
	if (p1 == NULL || p2 == NULL) {
		log("invalid ACTION 1 command \"%s\"", request);
		return;
	}
	*p2 = '\0';

	if (verify_path(p1) < 0) {
		log("access denied: %s", p1);
		return;
	}

	if (run_trigger(pretrigger, p1, reqsize - (p1 - request), tfilename, sizeof(tfilename)) < 0) {
		log("pretrigger %s failed", p1);
  		do_send(sock, "404", 3);
		return;
	}
	p1 = tfilename;

	if ((fd = open(p1, O_RDONLY)) < 0) {
		log("open %s: %s", p1, strerror(errno));
  		do_send(sock, "404", 3);
		run_trigger(posttrigger, p1, reqsize - (p1 - request), NULL, 0);
		return;
	}

  	if (do_send(sock, "200", 3) < 0) {
		close(fd);
		run_trigger(posttrigger, p1, reqsize - (p1 - request), NULL, 0);
		return;
	}

	store_recent(p1);

	while (1) {
		char newreq[512];

		if ((len = do_recv(sock, newreq, sizeof(newreq))) < 0) {
			close(fd);
			run_trigger(posttrigger, p1, reqsize - (p1 - request), NULL, 0);
			return;
		}

		if (strstr(newreq, "SIZE") == newreq)
			handle_size(sock, newreq, 512);
		else if (strstr(newreq, "GET") == newreq)
			handle_get(sock, newreq, 512, fd);
		else
			log("unknown ACTION 1 command \"%s\"", newreq);
	}
}

void handle_request(int sock)
{
	while (1) {
		char request[512];
		int len;

		if ((len = do_recv(sock, request, sizeof(request))) < 0)
			return;

  		if (strstr(request, "LIST") == request)
			handle_list(sock, request, 512);
		else if (strstr(request, "ACTION 1") == request)
			handle_action1(sock, request, 512);
		else if (strstr(request, "ACTION 2") == request)
			handle_action1(sock, request, 512);
		else if (strstr(request, "SIZE") == request)
			handle_size(sock, request, 512);
		else if (strstr(request, "GET") == request)
			handle_get(sock, request, 512, -1);
		else
			log("unknown KiSS command \"%s\"", request);
	}
}

void handle_kmlrequest(int sock)
{
	char request[512];
	int len;
	char *ok1 = "<?xml version='1.0' encoding='UTF-8' standalone='yes'?>\n<KMLPAGE>\n<GOTO href='";
	char *ok2 = "' />\n</KMLPAGE>\n";
	char *error = "<html><head>\n<title>501 Method Not Implemented</title>\n</head><body>\n<h1>Method Not Implemented</h1>\n</body></html>\n";

	if ((len = do_recv(sock, request, sizeof(request))) < 0)
		return;

  	if (strstr(request, "GET /index.kml") == request) {
		do_send(sock, ok1, strlen(ok1));
		do_send(sock, kmlurl, strlen(kmlurl));
		do_send(sock, ok2, strlen(ok2));
	}
	else {
		do_send(sock, error, sizeof(error));
		log("unknown KiSS kml command \"%s\"", request);
	}
}

/*
 *   Name: store_recent
 *
 *   Description: Writes currently requested file to recently used
 *                direcory list.
 */
static int store_recent(char *filename)
{
	char recent[PATH_MAX];
	char target[PATH_MAX];
	char fpath[PATH_MAX];
	int n, num_files;
	struct dirent **namelist;

	if (!strncmp(filename, audiopath, strlen(audiopath)))
		snprintf(recent, PATH_MAX, "%s/.recent", audiopath);
	else if (!strncmp(filename, videopath, strlen(videopath)))
		snprintf(recent, PATH_MAX, "%s/.recent", videopath);
	else if (!strncmp(filename, picturepath, strlen(picturepath)))
		snprintf(recent, PATH_MAX, "%s/.recent", picturepath);

	if (mkdir(recent,  S_IRWXU | S_IROTH | S_IXOTH) < 0)
		if (errno != EEXIST) {
			log("Couldn't create directory %s", recent);
			return -1;
		}

	chdir(recent); /* scandir does not provide path info to sort */
	num_files = scandir(recent, &namelist, 0, timesort);

	if (num_files < 0)
		log("scandir(%s) failed with %s", recent, strerror(errno));
	else {
		struct dirent * names = *namelist;

		for (n = 0; n < num_files; n++, names++) {
			/* skip parent and current dir */
			if (!strcmp (names->d_name, ".") ||
			    !strcmp (names->d_name, "..")) {
				free(names);
				continue;
			}

			/* delete excess files */
			if (n >= max_recent_files) {
				logv("Delete oldest recent entry %s",
				     names->d_name);
				snprintf(fpath, PATH_MAX, "%s/%s",
					 recent, names->d_name);
				unlink(fpath);
				free(names);
			}
		}
		free(namelist);
	}

	if (!strncmp(filename, recent, strlen(recent)))
		readlink(filename, target, PATH_MAX);
	else
		strncpy(target, filename, PATH_MAX);

	snprintf(fpath, PATH_MAX, "%s/%s", recent, basename(target));

	/* make sure link is created anew */
	unlink(fpath);

	if (symlink(target, fpath) < 0)
		if (errno != EEXIST) {
			log("store_recent: cannot create symlink %s, error %s",
			    recent, strerror(errno));
			return -1;
		}

	return 0;
}
