#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <sys/socket.h>
#include <sys/mman.h>

#include "kissd.h"

#ifdef USE_INTERNAL_SENDFILE_MMAP
ssize_t internal_sendfile(int dst, int src, off_t *offset, size_t size) {
	ssize_t len = 0;
	void *sp;
	off_t paged_offset;
	size_t mapped_size, mapped_offset;
	size_t pagesize = 65536; /* getpagesize() */

	if (size <= 0)
		return 0;

	/* Calculate offset as a multiple of the page size */
	paged_offset = (*offset / pagesize) * pagesize;
	mapped_offset = *offset - paged_offset;
	mapped_size = size + mapped_offset;

	logv("internal_sendfile(%i,%i,%lld,%u): "
	     "Mapping %u bytes at %lld, to read %u bytes at +%u",
	     dst, src, *offset, size, mapped_size, paged_offset,
	     size, mapped_offset);

	/* Map from paged offset to end of chunk */
	if ((sp = mmap(NULL, mapped_size,
		       PROT_READ, MAP_PRIVATE | MAP_NORESERVE,
		       src, paged_offset)) != MAP_FAILED) {

		len = write(dst, sp + mapped_offset, size);

		logv("internal_sendfile(%i,%i,%lld,%u): "
		     "Sent %u bytes at memloc %p to %u "
		     "(tried to send %u bytes)",
		     dst, src, *offset, size, len, sp + mapped_offset,
		     dst, size);

		*offset += len;

		if (munmap(sp, mapped_size) != 0)
			log("munmap() failed: %s", strerror(errno));

	} else {
		log("mmap(NULL,%u,%d,%d,%d,%lld) failed: %s",
		    mapped_size, PROT_READ,
		    MAP_PRIVATE | MAP_NORESERVE, src, paged_offset,
		    strerror(errno));
		len = -1;
	}
	return len;
}
#endif

#ifdef USE_INTERNAL_SENDFILE_COPY
ssize_t internal_sendfile(int dst, int src, off_t *offset, size_t size) {
	ssize_t len = 0;
	static char buf[128 * 1024];

	if (size <= 0)
		return 0;

	/*
	 * Seek to file position, read into static buffer
	 * and send entire buffer
	 */
	if (size <= sizeof(buf)) {
		if ((len = pread(src, buf, size, *offset)) > 0) {
			len = write(dst, buf, len);
			*offset += len;
		} else
			if (len < 0)
				log("pread() failed: %s", strerror(errno));
	}
	else {
		log("internal_sendfile(%i,%i,%lld,%u): "
		    "Cannot send %u bytes - "
		    "Static internal buffer is only %u bytes.",
		    dst, src, *offset, size, size, sizeof(buf));
		len = -1;
	}
	return len;
}
#endif
