/* 
 * kissd - KiSS PC-Link Daemon
 *
 * Copyright (C) 2005 Stelian Pop <stelian@popies.net>
 *
 * Heavily based on kiss4lin,
 * Copyright (C) 2004 Jacob Kolding <dacobi@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <dirent.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/sendfile.h>

#include "kissd.h"

static int do_recv(int sock, char *buffer, int size) {
	int len;

	if ((len = recv(sock, buffer, size, 0)) < 0) {
		log("recv: %s", strerror(errno));
		return len;
	}
	if (len == 0)
		return -1;
	while (len > 0 && (buffer[len - 1] == '\r' || buffer[len - 1] == '\n'))
		len--;
	buffer[len] = '\0';

	if (v_opt)
		log("<-- [%s]", buffer);

	return len;
}

static int do_send(int sock, char *buffer, int size) {
	int len;

	if (v_opt)
		log("--> [%s]", buffer);

	if ((len = send(sock, buffer, size, 0)) < 0) 
		log("send: %s", strerror(errno));
	return len;
}

/*
 * Name: clean_pathname
 *
 * Description: Replaces unsafe/incorrect instances of:
 *  //[...] with /
 *  /./ with /
 *  /../ with / (technically not what we want, but browsers should deal
 *   with this, not servers)
 */
void clean_pathname(char *pathname)
{
	char *cleanpath, c;

	cleanpath = pathname;
	while ((c = *pathname++)) {
		if (c == '/') { 
			while (1) {
				if (*pathname == '/')
					pathname++;
				else if (*pathname == '.' && *(pathname + 1) == '/')
					pathname += 2;
				else if (*pathname == '.' && *(pathname + 1) == '.' &&
					*(pathname + 2) == '/') {
					pathname += 3;
				} else      
					break;
			}               
		c = '/';
		}           
		*cleanpath++ = c;
	}
	*cleanpath = '\0';
}

static int verify_path(char *p)
{
	char path[PATH_MAX];

	snprintf(path, PATH_MAX, "/%s", p);
	clean_pathname(path);
	if (strstr(path, audiopath) == path ||
	    strstr(path, videopath) == path ||
	    strstr(path, picturepath) == path)
		return 0;
	return -1;
}

static void list_folder(int sock, char *base, char *path)
{
	char fullpath[PATH_MAX], line[PATH_MAX];
	int i, n;
	struct stat st;
	struct dirent **namelist;

	snprintf(fullpath, PATH_MAX, "%s/%s", base, path);
	if (verify_path(fullpath) < 0) {
		log("access denied: %s", fullpath);
		return;
	}

	if ((n = scandir(fullpath, &namelist, 0, alphasort)) < 0) {
		log("scandir %s: %s", fullpath, strerror(errno));
		return;
	}

	for (i = 0; i < n; i++) {
		char fpath[PATH_MAX];
		snprintf(fpath, PATH_MAX, "%s/%s/%s", base, path, namelist[i]->d_name);
		if (stat(fpath, &st) < 0) {
			log("stat %s: %s", fpath, strerror(errno));
			continue;
		}
		if (S_ISDIR(st.st_mode)) {
			if ((strcmp(namelist[i]->d_name, ".") == 0) ||
			    (strcmp(namelist[i]->d_name, "..") == 0))
				continue;
			snprintf(line, PATH_MAX, "%s|%s|1|\n", 
				namelist[i]->d_name, namelist[i]->d_name);
			do_send(sock, line, strlen(line));
		}
	}
	
	for (i = 0; i < n; i++) {
		char fpath[PATH_MAX];
		snprintf(fpath, PATH_MAX, "%s/%s/%s", base, path, namelist[i]->d_name);
		if (stat(fpath, &st) < 0) {
			log("stat %s: %s", fpath, strerror(errno));
			continue;
		}
		if (! S_ISDIR(st.st_mode)) {
			snprintf(line, PATH_MAX, "%s|%s|0|\n",
				namelist[i]->d_name, fpath);
			do_send(sock, line, strlen(line));
		}
		free(namelist[i]);
	}

	free(namelist);

	do_send(sock, "EOL\n", 4);
}

static void handle_list(int sock, char *request)
{
	char *p1, *p2;

	p1 = index(request, '|');
	p2 = rindex(request, '|');

	if (p1 == NULL || p2 == NULL) {
		log("invalid LIST command \"%s\"", request);
		return;
	}
	*p2 = '\0';

	if (strstr(request, "AUDIO") == request + 5)
		list_folder(sock, audiopath, p1 + 1);
	else if (strstr(request, "VIDEO") == request + 5)
		list_folder(sock, videopath, p1 + 1);
	else if (strstr(request, "PICTURE") == request + 5)
		list_folder(sock, picturepath, p1 + 1);
	else
    		log("unknown LIST command \"%s\"", request);
}

static void handle_size(int sock, char *request)
{
	struct stat st;
	char response[20];
	char *p1;

	p1 = index(request, '|');
	if (p1 == NULL) {
		log("invalid SIZE command \"%s\"", request);
		return;
	}
	*p1 = '\0';
	
	if (verify_path(request + 5) < 0) {
		log("access denied: %s", request + 5);
		return;
	}

	if (stat(request + 5, &st) < 0) {
		log("stat %s: %s", request + 5, strerror(errno));
      		return;
	}
 
 	snprintf(response, sizeof(response), "%015ld", st.st_size);
  	do_send(sock, response, 15);
}

static void handle_get(int sock, char *request, int fd)
{
	char *filename;
	char *p1, *p2;
	off_t offset;
	size_t chunk;
	int lfd = -1;
	long o, c;
  
	p1 = index(request, '/');
	p2 = index(request, '|');
	if (p1 == NULL || p2 == NULL) {
		log("invalid GET command \"%s\"", request);
		return;
	}
	*p2 = '\0';

	filename = p1;
	if (sscanf(p2 + 1, "%ld %ld", &o, &c) != 2) {
		log("invalid GET parameters \"%s\"", p2 + 1);
		return;
	};
	offset = o;
	chunk = c;

	if (fd == -1) {
		if (verify_path(filename) < 0) {
			log("access denied: %s", filename);
			return;
		}

		if ((lfd = open(filename, O_RDONLY)) < 0) {
			log("open %s: %s", filename, strerror(errno));
			return;
		}
		fd = lfd;
	}
  
	if (chunk == 0) {
		struct stat st;

		if (verify_path(filename) < 0) {
			log("access denied: %s", filename);
			return;
		}

		if (stat(filename, &st) < 0) {
			log("stat %s: %s", filename, strerror(errno));
      			if (lfd != -1)
				close(lfd);
      			return;
		}
		chunk = st.st_size;
	}

	if (sendfile(sock, fd, &offset, chunk) < 0) {
		log("sendfile: %s", strerror(errno));
		if (lfd != -1)
			close(lfd);
		return;
	}
  
	if (lfd != -1)
		close(lfd);
}

static void handle_action1(int sock, char *request)
{
	char *p1, *p2;
	int fd;
	int len;
  
	p1 = index(request, '/');
	p2 = rindex(request, '|');
	if (p1 == NULL || p2 == NULL) {
		log("invalid ACTION 1 command \"%s\"", request);
		return;
	}
	*p2 = '\0';
  
	if (verify_path(p1) < 0) {
		log("access denied: %s", p1);
		return;
	}

	if ((fd = open(p1, O_RDONLY)) < 0) {
		log("open %s: %s", p1, strerror(errno));
  		do_send(sock, "404", 3);
		return;
	}

  	if (do_send(sock, "200", 3) < 0)
		return;

	while (1) {
		char newreq[512];

		if ((len = do_recv(sock, newreq, sizeof(newreq))) < 0) {
			close(fd);
			return;
		}

		if (strstr(newreq, "SIZE") == newreq)
			handle_size(sock, newreq);
		else if (strstr(newreq, "GET") == newreq)
			handle_get(sock, newreq, fd);
		else
			log("unknown ACTION 1 command \"%s\"", newreq);
	}
}

void handle_request(int sock)
{
	char request[512];
	int len;

	if ((len = do_recv(sock, request, sizeof(request))) < 0)
		return;

  	if (strstr(request, "LIST") == request)
		handle_list(sock, request);
	else if (strstr(request, "ACTION 1") == request)
		handle_action1(sock, request);
	else if (strstr(request, "ACTION 2") == request)
		handle_action1(sock, request);
	else if (strstr(request, "SIZE") == request)
		handle_size(sock, request);
	else if (strstr(request, "GET") == request)
		handle_get(sock, request, -1);
	else
		log("unknown KiSS command \"%s\"", request);
}

void handle_kmlrequest(int sock)
{
	char request[512];
	int len;
	char *ok1 = "<?xml version='1.0' encoding='UTF-8' standalone='yes'?>\n<KMLPAGE>\n<GOTO href='";
	char *ok2 = "' />\n</KMLPAGE>\n";
	char *error = "<html><head>\n<title>501 Method Not Implemented</title>\n</head><body>\n<h1>Method Not Implemented</h1>\n</body></html>\n";

	if ((len = do_recv(sock, request, sizeof(request))) < 0)
		return;

  	if (strstr(request, "GET /index.kml") == request) {
		do_send(sock, ok1, strlen(ok1));
		do_send(sock, kmlurl, strlen(kmlurl));
		do_send(sock, ok2, strlen(ok2));
	}
	else {
		do_send(sock, error, strlen(error));
		log("unknown KiSS kml command \"%s\"", request);
	}
}
