/* 
 * Motion Eye video4linux application.
 *
 * Copyright (C) 2001-2002 Stelian Pop <stelian@popies.net>
 *
 * Copyright (C) 2001-2002 Alcve <www.alcove.com>
 *
 * Copyright (C) 2000 Andrew Tridgell <tridge@valinux.com>
 *
 * Earlier work by Werner Almesberger, Paul `Rusty' Russell and Paul Mackerras.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <getopt.h>
#include <linux/videodev.h>
#include <signal.h>

#include "motioneye.h"
#include <linux/meye.h>
#include <linux/sonypi.h>

static char *device = "/dev/video0";
static char *outfile = NULL;
static long captime = 10;
static int subsample = 0;
static int quality = 7;
static int brightness = 32;
static int hue = 32;
static int colour = 32;
static int contrast = 32;
static int sharpness = 32;
static int agc = 48;
static int picture = 0;
static int framerate = 0;
static char *version = "version 1.3, Jun 28, 2005";
static int spic_fd;
static int spic_on;
static int meye_capture = 1;

/* simple timing routines */
static struct timeval tp1,tp2;

static void timer_start(void) {
	gettimeofday(&tp1,NULL);
}

static double timer_end(void) {
	gettimeofday(&tp2,NULL);
	return((tp2.tv_sec - tp1.tv_sec) + 
	       (tp2.tv_usec - tp1.tv_usec)*1.0e-6);
}

static int init_meye(int *fd, unsigned char **mmapbuf, 
		     struct video_mbuf *mbuf, int *width, int *height) {

	struct video_picture pict;
	struct meye_params par;

	/* open the device */
	*fd = open(device, O_RDWR);
	if (*fd < 0) {
		perror("open");
		return 1;
	}

	/* set v4l picture properties */
	if (ioctl(*fd, VIDIOCGPICT, &pict) < 0) {
		perror("ioctl VIDIOCGPICT");
		return 1;
	}
	pict.brightness = (brightness << 10);
	pict.hue = (hue << 10);
	pict.colour = (colour << 10);
	pict.contrast = (contrast << 10);
	if (ioctl(*fd, VIDIOCSPICT, &pict) < 0) {
		perror("ioctl VIDIOCSPICT");
		return 1;
	}

	/* set extended picture/movie properties */
	if (ioctl(*fd, MEYEIOC_G_PARAMS, &par) < 0) {
		perror("ioctl MEYEIOC_G_PARAMS");
		return 1;
	}
	par.subsample = subsample;
	par.quality = quality;
	par.sharpness = sharpness;
	par.agc = agc;
	par.picture = picture;
	par.framerate = framerate;
	if (ioctl(*fd, MEYEIOC_S_PARAMS, &par) < 0) {
		perror("ioctl MEYEIOC_S_PARAMS");
		return 1;
	}

	/* mmap the device */
	if (ioctl(*fd, VIDIOCGMBUF, mbuf) < 0) {
		perror("ioctl VIDIOCGMBUF");
		return 1;
	}
	if ((*mmapbuf = mmap(0, mbuf->size, PROT_READ, MAP_SHARED, *fd, 0)) == MAP_FAILED) {
		perror("mmap");
		return 1;
	}

	/* set width/height */
	if (subsample) {
		*width = 320;
		*height = 240;
	}
	else {
		*width = 640;
		*height = 480;
	}
	
	return 0;
}

void cntlc(int sign) {
	meye_capture=0;
	printf("Received Cntl+C\n");
	return;
}

#ifdef WITH_X
static int capture(void) {
	int frames;
	char title[100]="";
	u8 rgb[640*480*3];
	int width;
	int height;
	int fd;
	int i;
	struct video_mbuf mbuf;
	struct video_mmap map;
	u8 *mmapbuf;
	sigset_t blockmask, oldmask, zeromask;

	if (init_meye(&fd, &mmapbuf, &mbuf, &width, &height))
		return 1;

	timer_start();

	frames = 0;

	meye_capture = 1;
	signal(SIGINT, cntlc);
	sigemptyset(&zeromask);
	sigemptyset(&blockmask);
	sigaddset(&blockmask, SIGINT);

	while (meye_capture) {
		for (i = 0; i < mbuf.frames; i++) {

			/* queue the frame */
			map.frame = i;
			map.width = width;
			map.height = height;
			map.format = VIDEO_PALETTE_YUV422;
			sigprocmask(SIG_BLOCK,&blockmask,&oldmask);
			if (ioctl(fd, VIDIOCMCAPTURE, &map) < 0) {
				perror("ioctl VIDIOCMCAPTURE");
				return 1;
			}

			/* get frame */
			if (ioctl(fd, VIDIOCSYNC, &i) < 0) {
				perror("ioctl VIDIOCSYNC");
				return 1;
			}
			sigprocmask(SIG_SETMASK, &oldmask, NULL);

			/* convert it */
			yuv_convert(mmapbuf + mbuf.offsets[i],
				    rgb, width, height);

			/* display the frame */
			frames++;
			if (frames % 5 == 0) {
				snprintf(title, sizeof(title)-1, 
					 "ViewFinder - %d fps",
					 (int)(0.5+frames/timer_end()));
			}
			display_rgb(rgb, width, height, title);

			if (timer_end() > 5) {
				frames=0;
				timer_start();
			}
		}
	}
	return 0;
}
#endif

static int ppmsnap(void) {
	u8 rgb[640*480*3];
	int width;
	int height;
	int fd;
	int i;
	struct video_mbuf mbuf;
	u8 *mmapbuf;
	FILE *f;

	if (init_meye(&fd, &mmapbuf, &mbuf, &width, &height))
		return 1;

	/* get the still */
	i = 0;
	if (ioctl(fd, MEYEIOC_STILLCAPT, 0) < 0) {
		perror("ioctl MEYEIOC_BUF_STILLCAPT");
		return 1;
	}

	/* convert it */
	yuv_convert(mmapbuf + mbuf.offsets[i], rgb, width, height);

	f = fopen(outfile,"w");
	if (!f) {
		perror("open");
		return 1;
	}
	fprintf(f,"P6\n%d %d\n255\n", width, height);
	fwrite(rgb, 3, width * height, f);
	fclose(f);

	close(fd);

	return 0;
}

static int jpgsnap(void) {
	int fd;
	int i;
	u8 *mmapbuf;
	FILE *f;
	int width, height;
	struct video_mbuf mbuf;

	if (init_meye(&fd, &mmapbuf, &mbuf, &width, &height))
		return 1;

	/* get the still */
	i = 0;
	if (ioctl(fd, MEYEIOC_STILLJCAPT, &i) < 0) {
		perror("ioctl MEYEIOC_BUF_STILLJCAPT");
		return 1;
	}

	f = fopen(outfile,"w");
	if (!f) {
		perror("open");
		return 1;
	}
	fwrite(mmapbuf, 1, i, f);
	fclose(f);

	close(fd);

	return 0;
}

static int mjpeg(void) {
	int fd;
	int i;
	u8 *mmapbuf;
	int f, len;
	struct video_mbuf mbuf;
	int width, height;
	int framecnt = 0;
	sigset_t blockmask, oldmask, zeromask;
	unsigned char event;

	if (init_meye(&fd, &mmapbuf, &mbuf, &width, &height))
		return 1;

	if (spic_fd != -1 && spic_on == 1) {
		printf("Waiting ... Push the CAPTURE button to start capture!!\n\n");
		while( read(spic_fd, &event, sizeof(event)) ) {
			if (event == SONYPI_EVENT_CAPTURE_PRESSED ||
			    event == SONYPI_EVENT_CAPTURE_PARTIALPRESSED)
				break;
			sleep(1);
		}
		printf("Release the CAPTURE button to end capture!!\n");
	}
	f = open(outfile, O_CREAT | O_RDWR | O_TRUNC, 0666);
	if (f < 0) {
		perror("open");
		return 1;
	}
	avi_start(f);

	timer_start();
	/* queue all frames */
	for (i = 0; i < mbuf.frames; ++i)
		if (ioctl(fd, MEYEIOC_QBUF_CAPT, &i) < 0) {
			perror("ioctl MEYEIOC_QBUF_CAPT");
			return 1;
		}

	meye_capture = 1;
	signal(SIGINT, cntlc);
	sigemptyset(&zeromask);
	sigemptyset(&blockmask);
	sigaddset(&blockmask, SIGINT);

	while (meye_capture) {
		for (i = 0; i < mbuf.frames; i++) {

			if (timer_end() > captime) 
				goto out;

			sigprocmask(SIG_BLOCK, &blockmask, &oldmask);
			/* get frame */
			len = i;
			if (ioctl(fd, MEYEIOC_SYNC, &len) < 0) {
				perror("ioctl MEYEIOC_SYNC");
				return 1;
			}

			/* re queue the frame */
			if (ioctl(fd, MEYEIOC_QBUF_CAPT, &i) < 0) {
				perror("ioctl MEYEIOC_QBUF_CAPT");
				return 1;
			}
			sigprocmask(SIG_SETMASK,&oldmask,NULL);

			avi_add(f, mmapbuf + mbuf.offsets[i], len);
			++framecnt;
			if (spic_on) {
				read(spic_fd, &event, sizeof(event));
				if (event == SONYPI_EVENT_CAPTURE_RELEASED)
					meye_capture=0;
			}
		}
	}

out:
	avi_end(f, width, height, (int)((double)framecnt / timer_end()));
	printf("%d frames captured...\n",framecnt);

	close(f);

	close(fd);
	return 0;
}

static void usage(char *program) {
	fprintf(stderr,
		"Motion Eye Camera Utility %s\n\n"
		"Usage: %s [COMMAND] [OPTION]...\n"
		"\nCommands:\n"
#ifdef WITH_X
		"\t-d, --display\t\tDisplay camera capture\n"
#endif
		"\t-p, --ppm=FILE\t\tGet camera snapshot (ppm format)\n"
		"\t-j, --jpg=FILE\t\tGet camera snapshot (jpg format)\n"
		"\t-m, --mjpeg=FILE\tGet mjpeg video\n"
		"\nOptions:\n"
		"\t-a, --agc=NUM\t\tCamera AGC (0-63) (default 48)\n"
		"\t-b, --brightness=NUM\tCamera brightness (0-63) (default 32)\n"
		"\t-c, --colour=NUM\tCamera colour (0-63) (default 32)\n"
		"\t-C, --contrast=NUM\tCamera contrast (0-63) (default 32)\n"
		"\t-D, --device=FILE\tVideo device to use (default /dev/video0)\n"
		"\t-f, --framerate=NUM\tFramerate (0=every frame, 2=every 2 frames) (0-31) (default 0)\n"
		"\t-h, --hue=NUM\t\tCamera hue (0-63) (default 32)\n"
		"\t-P, --picture=NUM\tCamera picture (0-63) (default 0)\n"
		"\t-q, --quality=NUM\tJPEG quality (1-10) (default 7)\n"
		"\t-s, --subsample\t\tSubsample the image\n"
		"\t-S, --sharpness=NUM\tCamera sharpness (0-63) (default 32)\n"
		"\t-t, --time=NUM\t\tNumber of seconds to capture (default 10)\n"
		"\t-w, --wait\t\tWait for the start capture with CAPTURE button\n"
		"\n", version, program);
}

int main(int argc, char *argv[]) {
	int option_index = 0;
	int c, choice = 0;
	static struct option long_options[] = {
		{"agc", 1, 0, 'a'},
		{"brightness", 1, 0, 'b'},
		{"colour", 1, 0, 'c'},
		{"contrast", 1, 0, 'C'},
#ifdef WITH_X
		{"display", 0, 0, 'd'},
#endif
		{"device", 0, 0, 'D'},
		{"framerate", 1, 0, 'f'},
		{"hue", 1, 0, 'h'},
		{"jpg", 1, 0, 'j'},
		{"mjpeg", 1, 0, 'm'},
		{"ppm", 1, 0, 'p'},
		{"picture", 1, 0, 'P'},
		{"quality", 1, 0, 'q'},
		{"subsample", 0, 0, 's'},
		{"sharpness", 1, 0, 'S'},
		{"time", 1, 0, 't'},
		{"wait", 0, 0, 'w'},
		{0, 0, 0, 0}
	};

	while (1) {
		c = getopt_long(argc, argv, "a:b:c:C:"
#ifdef WITH_X
				            "d"
#endif
					    "D:f:h:j:m:p:P:q:sS:t:w", 
					    long_options, &option_index);
		if (c == -1)
			break;

		switch (c) {
			case 'a':
				agc = strtoul(optarg, NULL, 0);
				if (agc < 0 || agc > 63) {
					fprintf(stderr, "agc must be between 0 and 63\n");
					exit(1);
				}
				break;
			case 'b':
				brightness = strtoul(optarg, NULL, 0);
				if (brightness < 0 || brightness > 63) {
					fprintf(stderr, "brightness must be between 0 and 63\n");
					exit(1);
				}
				break;
			case 'c':
				colour = strtoul(optarg, NULL, 0);
				if (colour < 0 || colour > 63) {
					fprintf(stderr, "colour must be between 0 and 63\n");
					exit(1);
				}
				break;
			case 'C':
				contrast = strtoul(optarg, NULL, 0);
				if (contrast < 0 || contrast > 63) {
					fprintf(stderr, "contrast must be between 0 and 63\n");
					exit(1);
				}
				break;
#ifdef WITH_X
			case 'd':
				choice = c;
				break;
#endif
			case 'D':
				device = optarg;
				break;
			case 'f':
				framerate = strtoul(optarg, NULL, 0);
				if (framerate < 0 || framerate > 31) {
					fprintf(stderr, "framerate must be between 0 and 31\n");
					exit(1);
				}
				break;
			case 'h':
				hue = strtoul(optarg, NULL, 0);
				if (hue < 0 || hue > 63) {
					fprintf(stderr, "hue must be between 0 and 63\n");
					exit(1);
				}
				break;
			case 'P':
				picture = strtoul(optarg, NULL, 0);
				if (picture < 0 || picture > 63) {
					fprintf(stderr, "picture must be between 0 and 63\n");
					exit(1);
				}
				break;
			case 'q':
				quality = strtoul(optarg, NULL, 0);
				if (quality < 0 || quality > 10) {
					fprintf(stderr, "quality must be between 0 and 10\n");
					exit(1);
				}
				break;
			case 's':
				subsample = 1;
				break;
			case 'S':
				sharpness = strtoul(optarg, NULL, 0);
				if (sharpness < 0 || sharpness > 63) {
					fprintf(stderr, "sharpness must be between 0 and 63\n");
					exit(1);
				}
				break;
			case 't':
				captime = strtoul(optarg, NULL, 0);
				break;
			case 'j':
			case 'm':
			case 'p':
				choice = c;
				outfile = optarg;
				break;
			case 'w':
				spic_on = 1;
				spic_fd = open("/dev/sonypi", O_RDONLY|O_NONBLOCK);
				break;
			default:
				fprintf(stderr, "unknown option %c\n", c);
				break;
		}
	}

	switch (choice) {
#ifdef WITH_X
		case 'd':
			return capture();
			break;
#endif
		case 'p':
			return ppmsnap();
			break;
		case 'j':
			return jpgsnap();
			break;
		case 'm':
			return mjpeg();
			break;
		default:
			usage(argv[0]);
			exit(1);
	}
	if (spic_fd != -1) 
		close(spic_fd);
	exit(0);
}
